<?php
/**
 * @author Jegtheme
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'JNews_Migration_Publisher' ) ) {
	class JNews_Migration_Publisher {

		/**
		 * @var JNews_Migration_Publisher
		 */
		private static $instance;

		/**
		 * @return JNews_Migration_Publisher
		 */
		public static function getInstance() {
			if ( null === static::$instance ) {
				static::$instance = new static();
			}
			return static::$instance;
		}

		/**
		 * JNews_Migration_Publisher constructor
		 */
		private function __construct() {
			add_action( 'wp_ajax_jnews_content_migration_publisher', [ $this, 'content_migration' ] );
			add_action( 'wp_ajax_nopriv_jnews_content_migration_publisher', [ $this, 'content_migration' ] );

			add_filter( 'jnews_get_total_view', [ $this, 'get_total_view' ], 99, 2 );
		}

		/**
		 * Main function for content migration
		 *
		 * @param  boolean $count ( set true if you want to return the number of post only )
		 */
		public function content_migration( $count = false ) {
			$posts = get_posts( $this->build_args( $count ) );

			if ( $count ) {
				return count( $posts );
			}

			if ( ! empty( $posts ) && is_array( $posts ) && check_admin_referer( 'jnews_migration_publisher', 'nonce' ) ) {
				foreach ( $posts as $post ) {
					$post_review = get_post_meta( $post->ID, '_bs_review_enabled', true );
					$post_video  = get_post_meta( $post->ID, '_featured_embed_code', true );

					if ( ! empty( $post_review ) ) {
						$this->review_migration_handler( $post->ID );
					}

					if ( ! empty( $post_video ) ) {
						set_post_format( $post->ID, 'video' );
						update_post_meta( $post->ID, '_format_video_embed', $post_video );
					}

					$this->post_layout_handler( $post );
					$this->post_category_handler( $post );

					$this->after_migration( $post );
				}
			}
		}

		public function post_category_handler( $post ) {
			$primary_category = get_post_meta( $post->ID, '_bs_primary_category', true );

			update_post_meta( $post->ID, 'jnews_primary_category', [ $primary_category ] );
		}

		public function post_layout_handler( $post ) {
			$options		= get_option( 'bs_publisher_theme_options' );
			$post_layout 	= get_post_meta( $post->ID, 'page_layout', true );
			$post_template 	= get_post_meta( $post->ID, 'post_template', true );
			$author_box		= isset( $options['post_author_box'] ) ? $options['post_author_box'] : 'show';
			$show_featured	= true;

			switch ( $post_layout ) {
				case '1-col':
					$post_layout = 'no-sidebar-narrow';
					break;

				case 'default':
				case '2-col-right':
					$post_layout = 'right-sidebar';
					break;

				case '2-col-left':
					$post_layout = 'left-sidebar';
					break;
			}

			switch ( $post_template ) {
				case 'style-9':
					$show_featured	= false;
					break;
			}

			switch ( $author_box ) {
				case 'hide':
					$author_box	= false;
					break;
				default:
					$author_box	= true;
					break;
			}

			$post_related = get_post_meta( $post->ID, 'post_related', true );

			$post_setting = [
				'override_template' => true,
				'override'          => [
					[
						'layout'                 => $post_layout,
						'paralax'                => true,
						'sidebar'                => 'default-sidebar',
						'show_featured'          => $show_featured,
						'show_share_counter'     => true,
						'show_view_counter'      => true,
						'show_post_meta'         => true,
						'show_post_author'       => $author_box,
						'show_post_author_image' => true,
						'show_post_date'         => true,
						'show_post_category'     => true,
						'show_post_tag'          => true,
						'show_prev_next_post'    => true,
						'show_popup_post'        => true,
						'show_post_related'      => ( $post_related == 'show' ) ? true : false,
					],
				],
			];

			update_post_meta( $post->ID, 'jnews_single_post', $post_setting );
		}

		/**
		 * Handler function for review post migration
		 *
		 * @param  int   $post_id
		 * @param  array $post_setting
		 */
		public function review_migration_handler( $post_id ) {
			update_post_meta( $post_id, 'enable_review', true );

			$review_type = get_post_meta( $post_id, '_bs_review_rating_type', true );
			$review_type = $this->get_type( $review_type );
			update_post_meta( $post_id, 'type', $review_type );

			update_post_meta( $post_id, 'name', get_post_meta( $post_id, '_bs_review_heading', true ) );

			$review_value = get_post_meta( $post_id, '_bs_review_criteria', true );
			$review_value = $this->get_rating( $review_value );
			update_post_meta( $post_id, 'rating', $review_value );

			$review_mean = $this->get_mean( $review_value );
			update_post_meta( $post_id, 'jnew_rating_mean', $review_mean );

			$review_summary = get_post_meta( $post_id, '_bs_review_verdict_summary', true );
			update_post_meta( $post_id, 'summary', $review_summary );

			update_post_meta( $post_id, 'jnews_review_fields', [ 'enable_review', 'type', 'name', 'summary', 'rating' ] );
		}

		/**
		 * Get review type
		 *
		 * @param  array $data
		 *
		 * @return string
		 */
		public function get_type( $data ) {
			$type = '';

			switch ( $data ) {
				case 'points':
					$type = 'point';
					break;

				case 'stars':
					$type = 'star';
					break;

				case 'percentage':
					$type = 'percentage';
					break;
			}

			return $type;
		}

		/**
		 * Get review rating
		 *
		 * @param  array $data
		 *
		 * @return array
		 */
		public function get_rating( $review_value ) {
			$rating = [];

			if ( is_array( $review_value ) ) {
				foreach ( $review_value as $point ) {
					$rating[] = [
						'rating_text'   => $point['label'],
						'rating_number' => $point['rate'],
					];
				}
			}

			return $rating;
		}

		/**
		 * Get review mean
		 *
		 * @param  array $data
		 *
		 * @return float ( default null )
		 */
		public function get_mean( $review_value ) {
			$total = $numberofrating = 0;

			if ( is_array( $review_value ) ) {
				foreach ( $review_value as $rating ) {
					if ( $rating['rating_number'] != 0 && ! empty( $rating['rating_text'] ) ) {
						$total += $rating['rating_number'];
						$numberofrating++;
					}
				}

				if ( $numberofrating > 0 ) {
					$mean = $total / $numberofrating;
					$mean = round( $mean, 1 );
					return $mean;
				}
			}

			return null;
		}

		/**
		 * Build query argument
		 *
		 * @param  boolean $count
		 *
		 * @return array
		 */
		public function build_args( $count ) {
			$args = [
				'post_type'      => 'post',
				'meta_query'     => [
					[
						'key'     => 'jnews_migration_publisher_status',
						'compare' => 'NOT EXISTS',
					],
					[
						'relation' => 'OR',
						[
							'key' => '_bs_primary_category',
						],
					],
				],
				'posts_per_page' => $count ? -1 : 1,
			];

			return $args;
		}

		/**
		 * End migration action
		 *
		 * @param  object $post
		 */
		public function after_migration( $post ) {
			update_post_meta( $post->ID, 'jnews_migration_publisher_status', true );

			$this->get_migration_response( $post );
		}

		/**
		 * Get migration response message
		 *
		 * @param  object $post
		 */
		public function get_migration_response( $post ) {
			wp_send_json(
				[
					'status'  => 'success',
					'message' => sprintf( __( 'Migration successful <strong>%1$s</strong> <a href="%2$s" target="_blank">view post</a>.', 'jnews-migration-publisher' ), $post->post_title, esc_url( get_permalink( $post->ID ) ) ),
				]
			);
		}

		/**
		 * Check JNews Review Plugin status
		 *
		 * @return false | string
		 */
		public function review_plugin_check() {
			if ( empty( $this->content_has_review() ) ) {
				 return false;
			}

			if ( ! function_exists( 'get_plugins' ) ) {
				require_once ABSPATH . '/wp-admin/includes/plugin.php';
			}

			if ( get_plugins( '/jnews-review' ) ) {
				if ( ! is_plugin_active( 'jnews-review/jnews-review.php' ) ) {
					return 'activate';
				}
			} else {
				return 'install';
			}

			return false;
		}

		/**
		 * Check if content has review post
		 *
		 * @return array
		 */
		public function content_has_review() {
			$args = [
				'post_type'      => 'post',
				'meta_query'     => [
					[
						'key'     => '_bs_review_enabled',
						'value'   => true,
						'compare' => '==',
					],
				],
				'posts_per_page' => -1,
			];

			$posts = get_posts( $args );

			return $posts;
		}

		public function get_total_view( $total, $post_id ) {
			$data = get_post_meta( $post_id, 'better-views-count', true );

			if ( ! empty( $data ) ) {
				$total = $total + (int) $data;
			}

			return $total;
		}
	}
}

